#!/bin/sh

. /lib/dracut-zfs-lib.sh

ZFS_DATASET=""
ZFS_POOL=""

case "${root}" in
	zfs:*) ;;
	*) return ;;
esac

GENERATOR_FILE=/run/systemd/generator/sysroot.mount
GENERATOR_EXTENSION=/run/systemd/generator/sysroot.mount.d/zfs-enhancement.conf

if [ -e "$GENERATOR_FILE" ] && [ -e "$GENERATOR_EXTENSION" ] ; then
	# If the ZFS sysroot.mount flag exists, the initial RAM disk configured
	# it to mount ZFS on root.  In that case, we bail early.  This flag
	# file gets created by the zfs-generator program upon successful run.
	info "ZFS: There is a sysroot.mount and zfs-generator has extended it."
	info "ZFS: Delegating root mount to sysroot.mount."
	# Let us tell the initrd to run on shutdown.
	# We have a shutdown hook to run
	# because we imported the pool.
	# We now prevent Dracut from running this thing again.
	for zfsmounthook in "$hookdir"/mount/*zfs* ; do
		if [ -f "$zfsmounthook" ] ; then
			rm -f "$zfsmounthook"
		fi
	done
	return
fi
info "ZFS: No sysroot.mount exists or zfs-generator did not extend it."
info "ZFS: Mounting root with the traditional mount-zfs.sh instead."

# Delay until all required block devices are present.
modprobe zfs 2>/dev/null
udevadm settle

if [ "${root}" = "zfs:AUTO" ] ; then
	ZFS_DATASET="$(find_bootfs)"
	if [ $? -ne 0 ] ; then
		zpool import -N -a ${ZPOOL_IMPORT_OPTS}
		ZFS_DATASET="$(find_bootfs)"
		if [ $? -ne 0 ] ; then
			warn "ZFS: No bootfs attribute found in importable pools."
			export_all -F

			rootok=0
			return 1
		fi
	fi
	info "ZFS: Using ${ZFS_DATASET} as root."
fi

ZFS_DATASET="${ZFS_DATASET:-${root#zfs:}}"
ZFS_POOL="${ZFS_DATASET%%/*}"

if import_pool "${ZFS_POOL}" ; then
	# Load keys if we can or if we need to
	if [ "$(zpool list -H -o feature@encryption "$(echo "${ZFS_POOL}" | awk -F/ '{print $1}')")" = 'active' ]; then
		# if the root dataset has encryption enabled
		ENCRYPTIONROOT="$(zfs get -H -o value encryptionroot "${ZFS_DATASET}")"
		if ! [ "${ENCRYPTIONROOT}" = "-" ]; then
			KEYSTATUS="$(zfs get -H -o value keystatus "${ENCRYPTIONROOT}")"
			# if the key needs to be loaded
			if [ "$KEYSTATUS" = "unavailable" ]; then
				# decrypt them
				ask_for_password \
					--tries 5 \
					--prompt "Encrypted ZFS password for ${ENCRYPTIONROOT}: " \
					--cmd "zfs load-key '${ENCRYPTIONROOT}'"
			fi
		fi
	fi
	# Let us tell the initrd to run on shutdown.
	# We have a shutdown hook to run
	# because we imported the pool.
	info "ZFS: Mounting dataset ${ZFS_DATASET}..."
	if mount_dataset "${ZFS_DATASET}" ; then
		ROOTFS_MOUNTED=yes
		return 0
	fi
fi

rootok=0
