#!/usr/bin/env bash
# shellcheck disable=SC2154

check() {
	# We depend on udev-rules being loaded
	[ "${1}" = "-d" ] && return 0

	# Verify the zfs tool chain
	for tool in "/sbin/zgenhostid" "/sbin/zpool" "/sbin/zfs" "/sbin/mount.zfs" ; do
		test -x "$tool" || return 1
	done

	return 0
}

depends() {
	echo udev-rules
	return 0
}

installkernel() {
	instmods zfs
	instmods zcommon
	instmods znvpair
	instmods zavl
	instmods zunicode
	instmods zlua
	instmods icp
	instmods spl
	instmods zlib_deflate
	instmods zlib_inflate
}

install() {
	inst_rules /lib/udev/rules.d/90-zfs.rules
	inst_rules /lib/udev/rules.d/69-vdev.rules
	inst_rules /lib/udev/rules.d/60-zvol.rules
	dracut_install hostid
	dracut_install grep
	dracut_install /sbin/zgenhostid
	dracut_install /sbin/zfs
	dracut_install /sbin/zpool
	# Workaround for https://github.com/openzfs/zfs/issues/4749 by
	# ensuring libgcc_s.so(.1) is included
	if ldd /sbin/zpool | grep -qF 'libgcc_s.so'; then
		# Dracut will have already tracked and included it
		:;
	elif command -v gcc-config >/dev/null 2>&1; then
		# On systems with gcc-config (Gentoo, Funtoo, etc.):
		# Use the current profile to resolve the appropriate path
		s="$(gcc-config -c)"
		dracut_install "/usr/lib/gcc/${s%-*}/${s##*-}/libgcc_s.so"*
	elif [ "$(echo /usr/lib/libgcc_s.so*)" != "/usr/lib/libgcc_s.so*" ]; then
		# Try a simple path first
		dracut_install /usr/lib/libgcc_s.so*
	elif [ "$(echo /lib*/libgcc_s.so*)" != "/lib*/libgcc_s.so*" ]; then
		# SUSE
		dracut_install /lib*/libgcc_s.so*
	else
		# Fallback: Guess the path and include all matches
		dracut_install /usr/lib*/gcc/**/libgcc_s.so*
	fi
	# shellcheck disable=SC2050
	if [ 0 -gt 0 ]; then
		for d in $libdirs; do
			[ -e "$d/" ] && dracut_install "$d/"
		done
	fi
	dracut_install /sbin/mount.zfs
	dracut_install /lib/udev/vdev_id
	dracut_install awk
	dracut_install cut
	dracut_install tr
	dracut_install head
	dracut_install /lib/udev/zvol_id
	inst_hook cmdline 95 "${moddir}/parse-zfs.sh"
	if [ -n "$systemdutildir" ] ; then
		inst_script "${moddir}/zfs-generator.sh" "$systemdutildir"/system-generators/dracut-zfs-generator
	fi
	inst_hook pre-mount 90 "${moddir}/zfs-load-key.sh"
	inst_hook mount 98 "${moddir}/mount-zfs.sh"
	inst_hook cleanup 99 "${moddir}/zfs-needshutdown.sh"
	inst_hook shutdown 20 "${moddir}/export-zfs.sh"

	inst_simple "${moddir}/zfs-lib.sh" "/lib/dracut-zfs-lib.sh"
	if [ -e /etc/zfs/zpool.cache ]; then
		inst /etc/zfs/zpool.cache
		type mark_hostonly >/dev/null 2>&1 && mark_hostonly /etc/zfs/zpool.cache
	fi

	if [ -e /etc/zfs/vdev_id.conf ]; then
		inst /etc/zfs/vdev_id.conf
		type mark_hostonly >/dev/null 2>&1 && mark_hostonly /etc/zfs/vdev_id.conf
	fi

	# Synchronize initramfs and system hostid
	if [ -f /etc/hostid ]; then
		inst /etc/hostid
		type mark_hostonly >/dev/null 2>&1 && mark_hostonly /etc/hostid
	elif HOSTID="$(hostid 2>/dev/null)" && [ "${HOSTID}" != "00000000" ]; then
		zgenhostid -o "${initdir}/etc/hostid" "${HOSTID}"
		type mark_hostonly >/dev/null 2>&1 && mark_hostonly /etc/hostid
	fi

	if dracut_module_included "systemd"; then
		mkdir -p "${initdir}/$systemdsystemunitdir/zfs-import.target.wants"
		for _service in "zfs-import-scan.service" "zfs-import-cache.service" ; do
			dracut_install "/usr/lib/systemd/system/$_service"
			if ! [ -L "${initdir}/$systemdsystemunitdir/zfs-import.target.wants/$_service" ]; then
				ln -sf ../$_service "${initdir}/$systemdsystemunitdir/zfs-import.target.wants/$_service"
				type mark_hostonly >/dev/null 2>&1 && mark_hostonly "/usr/lib/systemd/system/$_service"
			fi
		done

		inst "${moddir}"/zfs-env-bootfs.service "${systemdsystemunitdir}"/zfs-env-bootfs.service
		ln -s ../zfs-env-bootfs.service "${initdir}/${systemdsystemunitdir}/zfs-import.target.wants"/zfs-env-bootfs.service
		type mark_hostonly >/dev/null 2>&1 && mark_hostonly /usr/lib/systemd/system/zfs-env-bootfs.service

		dracut_install systemd-ask-password
		dracut_install systemd-tty-ask-password-agent

		mkdir -p "${initdir}/$systemdsystemunitdir/initrd.target.wants"
		dracut_install /usr/lib/systemd/system/zfs-import.target
		if ! [ -L "${initdir}/$systemdsystemunitdir/initrd.target.wants"/zfs-import.target ]; then
			ln -s ../zfs-import.target "${initdir}/$systemdsystemunitdir/initrd.target.wants"/zfs-import.target
			type mark_hostonly >/dev/null 2>&1 && mark_hostonly /usr/lib/systemd/system/zfs-import.target
		fi

		for _service in zfs-snapshot-bootfs.service zfs-rollback-bootfs.service ; do
			inst "${moddir}/$_service" "${systemdsystemunitdir}/$_service"
			if ! [ -L "${initdir}/$systemdsystemunitdir/initrd.target.wants/$_service" ]; then
				ln -s "../$_service" "${initdir}/$systemdsystemunitdir/initrd.target.wants/$_service"
			fi
		done

		# There isn't a pkg-config variable for this,
		# and dracut doesn't automatically resolve anything this'd be next to
		local systemdsystemenvironmentgeneratordir
		systemdsystemenvironmentgeneratordir="$(pkg-config --variable=prefix systemd || echo "/usr")/lib/systemd/system-environment-generators"
		mkdir -p "${initdir}/${systemdsystemenvironmentgeneratordir}"
		inst "${moddir}"/import-opts-generator.sh "${systemdsystemenvironmentgeneratordir}"/zfs-import-opts.sh
	fi
}
