#!/bin/sh
# shellcheck disable=SC2154

# only run this on systemd systems, we handle the decrypt in mount-zfs.sh in the mount hook otherwise
[ -e /bin/systemctl ] || [ -e /usr/bin/systemctl ] || return 0

# This script only gets executed on systemd systems, see mount-zfs.sh for non-systemd systems

# import the libs now that we know the pool imported
[ -f /lib/dracut-lib.sh ] && dracutlib=/lib/dracut-lib.sh
[ -f /usr/lib/dracut/modules.d/99base/dracut-lib.sh ] && dracutlib=/usr/lib/dracut/modules.d/99base/dracut-lib.sh
# shellcheck source=./lib-zfs.sh.in
. "$dracutlib"

# load the kernel command line vars
[ -z "$root" ] && root="$(getarg root=)"
# If root is not ZFS= or zfs: or rootfstype is not zfs then we are not supposed to handle it.
[ "${root##zfs:}" = "${root}" ] && [ "${root##ZFS=}" = "${root}" ] && [ "$rootfstype" != "zfs" ] && exit 0

# There is a race between the zpool import and the pre-mount hooks, so we wait for a pool to be imported
while [ "$(zpool list -H)" = "" ]; do
    systemctl is-failed --quiet zfs-import-cache.service zfs-import-scan.service && exit 1
    sleep 0.1s
done

# run this after import as zfs-import-cache/scan service is confirmed good
# we do not overwrite the ${root} variable, but create a new one, BOOTFS, to hold the dataset
if [ "${root}" = "zfs:AUTO" ] ; then
    BOOTFS="$(zpool list -H -o bootfs | awk '$1 != "-" {print; exit}')"
else
    BOOTFS="${root##zfs:}"
    BOOTFS="${BOOTFS##ZFS=}"
fi

# if pool encryption is active and the zfs command understands '-o encryption'
if [ "$(zpool list -H -o feature@encryption "${BOOTFS%%/*}")" = 'active' ]; then
    # if the root dataset has encryption enabled
    ENCRYPTIONROOT="$(zfs get -H -o value encryptionroot "${BOOTFS}")"
    if ! [ "${ENCRYPTIONROOT}" = "-" ]; then
        KEYSTATUS="$(zfs get -H -o value keystatus "${ENCRYPTIONROOT}")"
        # continue only if the key needs to be loaded
        [ "$KEYSTATUS" = "unavailable" ] || exit 0

        KEYLOCATION="$(zfs get -H -o value keylocation "${ENCRYPTIONROOT}")"
        case "${KEYLOCATION%%://*}" in
            prompt)
                for _ in 1 2 3; do
                    systemd-ask-password --no-tty "Encrypted ZFS password for ${BOOTFS}" | zfs load-key "${ENCRYPTIONROOT}" && break
                done
                ;;
            http*)
                systemctl start network-online.target
                zfs load-key "${ENCRYPTIONROOT}"
                ;;
            file)
                KEYFILE="${KEYLOCATION#file://}"
                [ -r "${KEYFILE}" ] || udevadm settle
                [ -r "${KEYFILE}" ] || {
                    info "Waiting for key ${KEYFILE} for ${ENCRYPTIONROOT}..."
                    for _ in 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20; do
                        sleep 0.5s
                        [ -r "${KEYFILE}" ] && break
                    done
                }
                [ -r "${KEYFILE}" ] || warn "Key ${KEYFILE} for ${ENCRYPTIONROOT} hasn't appeared. Trying anyway."
                zfs load-key "${ENCRYPTIONROOT}"
                ;;
            *)
                zfs load-key "${ENCRYPTIONROOT}"
                ;;
        esac
    fi
fi
